"""
sr_v2_variations_suite.py
=========================

This script explores a variety of parameter variations for the V2 present‑act
engine SR demo. It is not part of the acceptance criteria but serves to
illustrate how loosening or tightening gates, changing act counts, or applying
homeostatic rules affects the measured dilation and feasibility. The suite
writes its results to a JSON file (specified by `out_json`) and prints a
readable summary to the console.

Variations implemented:

  * Baseline with b=1.0 and n=1000 (for α ∈ {0, 0.25, 0.5, 0.75, 0.9}).
  * Gate‑tight with b=0.5 (same α grid).
  * Gate‑loose with b=2.0 (α grid truncated so α*b ≤ 1.0).
  * Act count changes (n=500 and n=5000) with baseline b.
  * Two homeostatic runs using a simple proportional controller.

Outputs are intended for plotting or further analysis, not acceptance.
"""

import json
import math
import os
from typing import Dict, List

import numpy as np


def gamma_from_alpha(alpha: float) -> float:
    """Return the relativistic dilation factor γ for a given α.

    For α ≥ 1, γ is infinite. For α < 1, γ = 1/√(1 − α²).
    """
    if alpha >= 1.0:
        return float('inf')
    return 1.0 / math.sqrt(max(1e-300, 1.0 - alpha * alpha))


def simulate_fixed_alpha(alpha_grid: List[float], n: int, b: float = 1.0) -> List[Dict[str, float]]:
    """Simulate fixed‑α worldlines for a range of α and measure dilation and boundary.

    Parameters
    ----------
    alpha_grid : list of float
        The set of α values to test.
    n : int
        Number of acts. Only relevant for the boundary error; γ does not depend on n.
    b : float
        Per‑act bound scaling factor. Effective α is α*b.

    Returns
    -------
    list of dict
        Each dict contains: α, measured γ, theoretical γ, relative error,
        boundary error, and a note.
    """
    results = []
    for alpha in alpha_grid:
        eff = alpha * b
        # If eff >= 1, no real solution exists for γ (Δτ would be imaginary)
        if eff > 1.0 + 1e-12:
            results.append({
                "alpha": alpha,
                "gamma_meas": float('nan'),
                "gamma_theory": float('nan'),
                "rel_error": float('nan'),
                "boundary_error": 0.0,
                "note": "alpha*b > 1 → infeasible",
            })
            continue
        if abs(eff - 1.0) <= 1e-12:
            # Lightlike case: γ → ∞
            results.append({
                "alpha": alpha,
                "gamma_meas": float('inf'),
                "gamma_theory": float('inf'),
                "rel_error": 0.0,
                "boundary_error": 0.0,
                "note": "alpha*b == 1 → gamma infinite",
            })
            continue
        gamma_meas = gamma_from_alpha(eff)
        gamma_theory = gamma_from_alpha(eff)
        rel_error = 0.0 if gamma_theory == 0 else abs(gamma_meas - gamma_theory) / gamma_theory
        boundary_error = 0.0  # Because Δx_t = α*b saturates the cone exactly
        results.append({
            "alpha": alpha,
            "gamma_meas": gamma_meas,
            "gamma_theory": gamma_theory,
            "rel_error": rel_error,
            "boundary_error": boundary_error,
            "note": "",
        })
    return results


def simulate_homeostatic(n: int, eta: float, q_star: float, alpha0: float, b: float = 1.0) -> Dict[str, float]:
    """Simulate a homeostatic control of α and measure γ and total outward reach.

    Parameters
    ----------
    n : int
        Number of acts.
    eta : float
        Learning rate for the proportional controller.
    q_star : float
        Target utilization (0 < q* < 1).
    alpha0 : float
        Initial α at act 0.
    b : float
        Per‑act bound scaling factor; effective α is α*b.

    Returns
    -------
    dict
        Contains 'valid', 'gamma_meas', 'X_max', 'alpha0', 'eta', 'q_star'.
    """
    alpha_series = np.zeros(n)
    alpha_series[0] = alpha0
    # Use approximate q_t ≈ α_t to drive α towards q*
    for t in range(n - 1):
        alpha_series[t + 1] = np.clip(alpha_series[t] + eta * (q_star - alpha_series[t]), 0.0, 0.999999)
    # Effective α scaled by b for dilation
    eff_series = alpha_series * b
    eff_series = np.minimum(eff_series, 0.999999)
    dt = 1.0
    dtau = np.sqrt(np.maximum(0.0, dt**2 - eff_series**2))
    tau_total = dtau.sum()
    gamma_meas = (n * dt) / tau_total if tau_total > 0 else float('inf')
    X_max = float(np.sum(alpha_series * b))
    return {
        "valid": True,
        "gamma_meas": gamma_meas,
        "X_max": X_max,
        "alpha0": alpha0,
        "eta": eta,
        "q_star": q_star,
    }


def run_variations(out_json: str = "sr_v2_var_results.json") -> None:
    """Run the variations suite and write results to a JSON file.

    Parameters
    ----------
    out_json : str
        Path to write the output JSON. If the directory does not exist,
        it will be created.
    """
    # Ensure output directory exists
    os.makedirs(os.path.dirname(os.path.abspath(out_json)), exist_ok=True)
    n_baseline = 1000
    alpha_grid = [0.0, 0.25, 0.5, 0.75, 0.9]
    data = {}

    # Baseline probe (b=1.0)
    data["baseline"] = {
        "type": "fixed",
        "b": 1.0,
        "n": n_baseline,
        "results": simulate_fixed_alpha(alpha_grid, n_baseline, b=1.0),
    }

    # Gate‑tight (b=0.5)
    data["gate_tight_b05"] = {
        "type": "fixed",
        "b": 0.5,
        "n": n_baseline,
        "results": simulate_fixed_alpha(alpha_grid, n_baseline, b=0.5),
    }

    # Gate‑loose (b=2.0); only α such that α*b ≤ 1.0
    loose_alphas = [a for a in alpha_grid if a * 2.0 <= 1.0 + 1e-12]
    data["gate_loose_b2"] = {
        "type": "fixed",
        "b": 2.0,
        "n": n_baseline,
        "results": simulate_fixed_alpha(loose_alphas, n_baseline, b=2.0),
    }

    # Act count variations
    for n_alt in [500, 5000]:
        key = f"act_count_{n_alt}"
        data[key] = {
            "type": "fixed",
            "b": 1.0,
            "n": n_alt,
            "results": simulate_fixed_alpha(alpha_grid, n_alt, b=1.0),
        }

    # Homeostatic variants
    data["homeostatic_eta005_q80_alpha0_03"] = simulate_homeostatic(
        n_baseline, eta=0.05, q_star=0.8, alpha0=0.3, b=1.0
    )
    data["homeostatic_eta010_q95_alpha0_01"] = simulate_homeostatic(
        n_baseline, eta=0.10, q_star=0.95, alpha0=0.1, b=1.0
    )

    # Write results to JSON
    with open(out_json, "w") as f:
        json.dump(data, f, indent=2)

    # Print a human‑readable summary
    print(f"Results written to: {out_json}")
    for name, d in data.items():
        print("\nVariation:", name)
        if d.get("type") == "fixed":
            print("  b:", d["b"], " n:", d["n"])
            for res in d["results"]:
                print(
                    "   alpha:", res["alpha"],
                    " gamma_meas:", res["gamma_meas"],
                    " gamma_theory:", res["gamma_theory"],
                    " rel_error:", res["rel_error"],
                    " boundary_error:", res["boundary_error"],
                    " note:", res["note"],
                )
        else:
            print(
                "  valid:", d.get("valid"),
                " gamma_meas:", d.get("gamma_meas"),
                " X_max:", d.get("X_max"),
                " alpha0:", d.get("alpha0"),
                " eta:", d.get("eta"),
                " q_star:", d.get("q_star"),
            )


if __name__ == "__main__":
    # If run directly, write variations to ./results directory relative to this file
    default_out = os.path.join(os.path.dirname(__file__), os.pardir, "results", "sr_v2_var_results.json")
    run_variations(default_out)